﻿namespace Microsoft.Samples.PlanMyNight.Data.Caching
{
    using System;
    using System.Collections.Generic;
    using System.Diagnostics;
    using System.Globalization;
    using Microsoft.Samples.PlanMyNight.Entities;

    public class CachedItinerariesRepository : IItinerariesRepository
    {
        private const string ItineraryCacheContainer = "Itinerary";

        private const string ItineraryCommentsCacheContainer = "ItineraryComments";

        private const string UsernameCacheContainer = "Reference";

        private readonly TimeSpan ItineraryCacheTimeOut = TimeSpan.FromMinutes(30);

        private readonly TimeSpan ItineraryCommentsCacheTimeOut = TimeSpan.FromMinutes(15);

        private readonly TimeSpan UsernameCacheTimeOut = TimeSpan.FromMinutes(10);

        private readonly ICachingProvider cacheProvider;

        private readonly IItinerariesRepository repository;

        public CachedItinerariesRepository(ICachingProvider cacheProvider) :
            this(cacheProvider, new ItinerariesRepository())
        {
        }

        public CachedItinerariesRepository(ICachingProvider cacheProvider, IItinerariesRepository repository)
        {
            this.cacheProvider = cacheProvider;
            this.repository = repository;
        }

        public PagingResult<Itinerary> SearchByActivity(string activityId, int pageSize, int pageNumber)
        {
            return this.repository.SearchByActivity(activityId, pageSize, pageNumber);
        }

        public PagingResult<Itinerary> SearchByZipCode(int activityTypeId, string zip, int pageSize, int pageNumber)
        {
            return this.repository.SearchByZipCode(activityTypeId, zip, pageSize, pageNumber);
        }

        public PagingResult<Itinerary> SearchByCity(int activityTypeId, string state, string city, int pageSize, int pageNumber)
        {
            return this.repository.SearchByCity(activityTypeId, state, city, pageSize, pageNumber);
        }

        public PagingResult<Itinerary> SearchByRadius(int activityTypeId, double longitude, double latitude, double radius, int pageSize, int pageNumber)
        {
            return this.repository.SearchByRadius(activityTypeId, longitude, latitude, radius, pageSize, pageNumber);
        }

        public Itinerary Retrieve(long itineraryId)
        {
            var cacheKey = itineraryId.ToString();

            var data = this.cacheProvider.Get(ItineraryCacheContainer, cacheKey) as Itinerary;
            if (data != null)
            {
                Trace.WriteLine("GET-CACHE:" + ItineraryCacheContainer + "(" + cacheKey + ")");
                return data;
            }

            data = this.repository.Retrieve(itineraryId);
            this.cacheProvider.Add(ItineraryCacheContainer, cacheKey, data, this.ItineraryCacheTimeOut);
            Trace.WriteLine("ADD-CACHE:" + ItineraryCacheContainer + "(" + cacheKey + ")");
            return data;
        }

        public IEnumerable<Itinerary> RetrieveByUser(Guid userId)
        {
            return this.repository.RetrieveByUser(userId);
        }

        public void Add(Itinerary itinerary)
        {
            this.repository.Add(itinerary);
        }

        public void Update(Itinerary itinerary)
        {
            this.repository.Update(itinerary);
            var cacheKey = itinerary.Id.ToString();
            this.cacheProvider.Remove(ItineraryCacheContainer, cacheKey);
        }

        public string GetUserDisplayName(Guid userId)
        {
            var cacheKey = string.Format(CultureInfo.InvariantCulture, "username_{0}", userId.ToString());

            var data = this.cacheProvider.Get(UsernameCacheContainer, cacheKey) as string;
            if (data != null)
            {
                Trace.WriteLine("GET-CACHE:" + UsernameCacheContainer + "(" + cacheKey + ")");
                return data;
            }

            data = this.repository.GetUserDisplayName(userId);
            this.cacheProvider.Add(UsernameCacheContainer, cacheKey, data, this.UsernameCacheTimeOut);
            Trace.WriteLine("ADD-CACHE:" + UsernameCacheContainer + "(" + cacheKey + ")");
            return data;
        }

        public bool CanUserRateItinerary(long itineraryId, Guid userId)
        {
            return this.repository.CanUserRateItinerary(itineraryId, userId);
        }

        public void RateItinerary(long itineraryId, Guid userId, byte rating, DateTime timestamp)
        {
            this.repository.RateItinerary(itineraryId, userId, rating, timestamp);
            this.cacheProvider.Remove(ItineraryCacheContainer, itineraryId.ToString());
        }

        public IEnumerable<ItineraryComment> RetrieveComments(long itineraryId)
        {
            var cacheKey = itineraryId.ToString();

            var data = this.cacheProvider.Get(ItineraryCommentsCacheContainer, cacheKey) as IEnumerable<ItineraryComment>;
            if (data != null)
            {
                Trace.WriteLine("GET-CACHE:" + ItineraryCommentsCacheContainer + "(" + cacheKey + ")");
                return data;
            }

            data = this.repository.RetrieveComments(itineraryId);
            this.cacheProvider.Add(ItineraryCommentsCacheContainer, cacheKey, data, this.ItineraryCommentsCacheTimeOut);
            Trace.WriteLine("ADD-CACHE:" + ItineraryCommentsCacheContainer + "(" + cacheKey + ")");
            return data;
        }

        public void AddComment(ItineraryComment comment)
        {
            this.repository.AddComment(comment);

            var cacheKey = comment.ItineraryId.ToString();
            var previousComments = this.cacheProvider.Get(ItineraryCommentsCacheContainer, cacheKey) as IEnumerable<ItineraryComment>;
            if (previousComments != null)
            {
                // update cache
                var list = new List<ItineraryComment>(previousComments);
                list.Add(comment);
                this.cacheProvider.Add(ItineraryCommentsCacheContainer, cacheKey, list.ToArray(), this.ItineraryCommentsCacheTimeOut);
            }
        }
    }
}
